;##########################################
;#####            PROTON              #####
;#####    PC-1 HEXADECIMAL MONITOR    #####
;#####          VERSION  3.0          #####
;##########################################
;
; KEYBOARD HAS INVERTED DRIVERS  25-08-1982
;
;
;---------------------------------------
;   ### COMMAND DESCRIPTION ### 
;
; DISPLAY NEXT/PREVIOUS LOCATION, ENTER ADDRESS
; GO TO USER'S PROGRAM
; INPUT ADDRESS
; INSERT 1 BYTE INTO THE PROGRAM
; DELETE 1 BYTE FROM THE PROGRAM
; CALCULATE BRANCH OFFSET
; SINGLE STEP INSTRUCTION(S)
; LOAD CODE FROM AUDIO-CASSETTE
; DUMP CODE TO CASSETTE
;
; INSTANT ASSEMBLER & DISASSEMBLER 
;
; THE MONITOR DRIVES A 16 CHARACTER DISPLAY,
; HANDLES INTERRUPTS, BREAKPOINTS AND INCORPORATES
; EIGHT 10-MILLISECOND TIMERS.
;
;
;---------------------------------------
;   ### SYSTEM ADDRESSES ### 
;
ROMBAS  .EQU $F000                      ; START OF PROGRAM
ROMSIZ  .EQU $1000                      ; 4K (E)PROM
IOSEL   .EQU $E000                      ; I/O SELECT
MONRAM  .EQU $F0
;
;---------------------------------------
;   ### VIA ADDRESSING ### 
;
; VIA #1
DRB1    .EQU IOSEL + $10
DRA1    .EQU IOSEL + $11
DDRB1   .EQU IOSEL + $12
DDRA1   .EQU IOSEL + $13
T1CL1   .EQU IOSEL + $14
T1CH1   .EQU IOSEL + $15
T1LL1   .EQU IOSEL + $16
T1LH1   .EQU IOSEL + $17
T2LL1   .EQU IOSEL + $18
T2CH1   .EQU IOSEL + $19
SR1     .EQU IOSEL + $1A
ACR1    .EQU IOSEL + $1B
PCR1    .EQU IOSEL + $1C
IFR1    .EQU IOSEL + $1D
IER1    .EQU IOSEL + $1E
APDAT   .EQU IOSEL + $1F
;
;
;---------------------------------------
;   ### ASSEMBLER CONSTANTS ### 
;
ENTER   .EQU $0D
ESC     .EQU $1B
;
; COMMAND-KEY VALUES
MEMCMD  .EQU $0C                        ; FORW.SPACE
MINCMD  .EQU $08                        ; BACK.SPACE
ADRCMD  .EQU $4D                        ; 'M' = MEMORY SHOW/ALTER
SSTCMD  .EQU $53                        ; 'S' = SINGLE-STEP A PROGRAM
GOCMD   .EQU $47                        ; 'G' = GO TO USER PROGRAM
LODCMD  .EQU $52                        ; 'R' = READ FROM CASSETTE
DMPCMD  .EQU $57                        ; 'W' = WRITE TO CASSETTE
OFFCMD  .EQU $4F                        ; 'O' = CALCULATE BRANCH-OFFSET
DISCMD  .EQU $4B                        ; 'K' = DISASSEMBLE
ASSCMD  .EQU $49                        ; 'I' = ASSEMBLE
REGCMD  .EQU $3F                        ; '?' = SHOW REGISTERS
INSCMD  .EQU $11                        ; 'CHAR INSERT' = INSERT A BYTE IN MEMORY
DELCMD  .EQU $12                        ; 'CHAR DELETE' = REMOVE A BYTE FROM MEMORY
TSTCMD  .EQU $14                        ; CTRL-T = KEYBOARD --> DISPLAY ECHO TEST
;
; HARDWARE CONSTANTS
NMICTR  .EQU IOSEL + $07                ; CONTROL-LATCH
;
CHSEL   .EQU IOSEL + $04
SEGM0   .EQU IOSEL + $03
SEGM1   .EQU IOSEL + $02
;
INKB    .EQU IOSEL + $06                ; KEYBOARD BUFFER
OUTKB   .EQU IOSEL + $05                ; KEYBOARD LATCH
;
;                                       ; ZERO PAGE
NMIVEC  .EQU MONRAM                     ; INDIRECT VECTORS
INTVEC  .EQU MONRAM + $02               ; IRQ BEFORE MONITOR
TEMP    .EQU MONRAM + $04               ; 
IRQVEC  .EQU MONRAM + $06               ; IRQ AFTER MONITOR, NO 'BRK'
ADL     .EQU MONRAM + $08               ; MEMORY POINTER
ADH     .EQU MONRAM + $09               ; 
AC      .EQU MONRAM + $0A               ; USER REG A
XR      .EQU MONRAM + $0B               ;          X
YR      .EQU MONRAM + $0C               ;          Y
PS      .EQU MONRAM + $0D               ; USER STATUS
SPTR    .EQU MONRAM + $0E               ; USER STACKPOINTER
PRVCMD  .EQU MONRAM + $0F               ; PREVIOUS COMMAND
;
;---------------------------------------
;   ### FROM HERE IN PAGE 1 ### 
;
ID      .EQU $0100                      ; ID NUMBER FOR CASSETTE
TAPID   .EQU $0101                      ; ID. OF CURRENT FILE
EAL     .EQU $0102                      ; END OF EDITING BUFFER
EAH     .EQU $0103                      ; 
DIV1    .EQU $0104                      ; 
CURCMD  .EQU $0105                      ; CURRENT COMMAND
GKX     .EQU $0106                      ; AUX SAVE .I
TIMER   .EQU $0107                      ; IO MSEC TIMER (DECR.)
DBCNTR  .EQU $0117                      ; KEYBOARD DEBOUNCE
ASAV    .EQU $0119                      ; SAVE: A
XSAV    .EQU $011A                      ;       X
YSAV    .EQU $011B                      ;       Y
INDADR  .EQU $011C                      ; JUMP FOR COMMANDO'S
INDADL  .EQU $011C                      ; 
INDADH  .EQU $011D                      ; 
CHKSML  .EQU $011E                      ; CHECKSUM
CHKSMH  .EQU $011F                      ; 
;
DIBUFL  .EQU 16                         ; DISPLAYBUFFER SIZE
DIBUFF  .EQU $0120                      ; DISPLAY REFRESH BUFFER
DCPTR   .EQU $0130                      ; DISPLAY POINTER
OMASK   .EQU $0131                      ; MASK FOR KEYBOARD-OUTPUT
IMASK   .EQU $0132                      ; MASK FOR INPUT-DECODE
KEYNR   .EQU $0133                      ; LOGICAL NUMBER OF THE KEY
LSTKEY  .EQU $0134                      ; ASCII VALUE OF CURRENT & LAST KEY
CPATRN  .EQU $0135                      ; PATTERN FOR CONTROL-KEYS
REPTO   .EQU $0136                      ; REPEAT-TIME COUNTER 
CURPOS  .EQU $0137                      ; DISPLAY CURSOR POSITION
OXSAV   .EQU $0138                      ; X-SAV FOR OUTALL
;
.ORG ROMBAS
;
;---------------------------------------
;   ### VECTOR_LIST FOR USER_ENTRIES ### 
;
        JMP MONITR
        JMP GETKEY
        JMP SCNKEY
        JMP CLRDSP
        JMP WBYTE
        JMP RBYTE
        JMP WCHAR
        JMP RCHAR
        JMP TPRINT
        JMP BLKDIS
        JMP RELDIS
        JMP DUMP
        JMP LOAD
        JMP OUTALL
;
;---------------------------------------
;   ### RESET-ENTRY ###
;
;   INITIALIZE WORK-SPACE IF COLD RESET
;
RESET:  LDX #$FF                        ; INITIAL STACK VALUE
        TXS
        SEI                             ;  AND BLOCK INTERRUPTS
        JSR BLKDIS                      ; BLOCK DISPLAY-INTERRUPTS
        STX SPTR                        ; USER STACK
        CLD                             ; BINARY MODE
; TEST FOR WARM RESTART        
        LDA NMIVEC
        CMP VECTAB
        BNE RS3A
        LDA NMIVEC + 1
        CMP VECTAB + 1
        BEQ WARM                        ; WARM RESTART
; COLD START
RS3A:   LDY #VECLEN
RS1:    LDA VECTAB,Y
        STA MONRAM,Y
        DEY
        BPL RS1
        LDA #$00
        LDX #TIMER + 1 - EAH
RS2:    STA EAH,X
        DEX
        BPL RS2
        LDA #$FF
        STA EAL
        LDA #$06
        STA EAH
        LDA #$01
        STA ID                          ; DEFAULT = 1
        STA DIV1
        CLI                             ; ENABLE INTERRUPTS
        PHP
        PLA
        STA PS                          ; SETUP A VALID PROGRAM-STATUS 
;        
WARM:   LDY #(HDRMSG/256)               ; PRINT HEADER-MESSAGE
        LDA #(HDRMSG&$FF)
        JSR TPRINT
        JSR VIAINI                      ; RELEASE PB7 BY CB2=LOW
;
;---------------------------------------
;   ### MONITOR COMMAND LOOP ###
;
; GET A COMMAND FROM THE KEYBOARD AND
; START THE APROPRIATE COMMAND FUNCTION ROUTINE.
;
MONITR: LDX SPTR
        TXS
        JSR RELDIS                      ; RELEASE DISPLAY
        CLD
        JSR GETUKY                      ; GET UPPER-CASE KEY
JUMPER: LDX SPTR                        ; SET SYSTEM STACK
        TXS
        STA CURCMD
        LDX #NCMNDS
JP1:    CMP CMDTAB,X
        BEQ JP2
        DEX
        BPL JP1
        LDA #$3F
        STA CURCMD                       ; ILLEGAL COMMAND
        JSR WCOM1
        JMP MONITR
JP2:    TXA
        ASL A
        TAX
        LDA JMPTAB,X
        STA INDADL
        LDA JMPTAB + 1,X
        STA INDADH
        JSR CMD
        JMP MONITR
CMD:    JMP (INDADR)
;
;---------------------------------------
; THIS TABLE CONTAINS THE COMMAND KEY-VALUES
; (THE COMMAND-CHAR CODES).
;
CMDTAB: .DB MEMCMD, MINCMD, ADRCMD, SSTCMD
        .DB GOCMD, INSCMD, DELCMD, LODCMD
        .DB DMPCMD, OFFCMD, TSTCMD, DISCMD
        .DB ASSCMD, REGCMD
NCMNDS  .EQU $ - CMDTAB -1
;
;---------------------------------------
; THIS TABLE CONTAINS THE ADDRESSES OF THE
; COMMAND FUNCTIONS ROUTINES.
; THE ROUTINES MUST RETURN WITH 'RTS'
;
JMPTAB: .DW PMEM, MMEM, INPADR, SSUSER
        .DW GOUSER, INSDEL, INSDEL, LOAD
        .DW DUMP, OFFSET, TESTER, DISA
        .DW $FB96, RSHOW                ; $FB96 = 'MNEENT' IN ASSEMBLER ROUTINES
;
;---------------------------------------
; DATA TO SETUP THE MONITOR WORKSPACE
;
VECTAB: .DW TERMNL, IRQHDL, NSUP, $0000
        .DW $0200                       ; PROGRAM COUNTER ?
        .DB $00, $00, $00, $00          ; REGISTERS 
VECLEN  .EQU $ - VECTAB - 1
;
HDRMSG: .DB "PROTON PC-1 V3-0", 0
;
;---------------------------------------
;   ### ENTRY FROM SINGLE-STEP ###
;
; EXIT THE USERPROGRAM BY AN 'IRQ' AND
; ENTER THE MONITOR.
;
SSTINT: STA AC
        STX XR
        STY YR
        PLA                             ; STATUS
        STA PS
        PLA                             ; PROGRAM COUNTER
        STA ADL
        PLA
        STA ADH
        TSX                             ; RP
        STX SPTR
        LDA T2LL1                       ; CLEAR SST-INT
        CLI
        CLD
        JSR RSHOW                       ; SHOW REGISTERSET
        JSR GETUKY                      ; READ A NEW COMMAND
        CMP #SSTCMD
        BNE TOJMP
        JMP SSUSER                      ; GO STEP AGAIN
TOJMP:  JMP JUMPER
;
;---------------------------------------
;   ### DIRECT TO +MEM & -MEM ###
;
PMEM:   LDA #$3E
        STA CURCMD
        BNE SEQADR
MMEM:   LDA #$3C
        STA CURCMD
;
;---------------------------------------
;   ### INPUT DATA ON SEQ. ADDRESSES ###
;
; READ THE ADDRESS FROM THE DISPLAY
; AND WRITE THE MEMORYCONTENTS INTO
; THE DATA-FIELD
;
SEQADR: JSR WCOMND                      ; CLEAR DISPLAY & WRITE COMMAND
        LDX #$00
        LDA ADH
        JSR WBYTE
        LDA ADL
        JSR WBYTE
SQA1:   LDX #$00
        JSR RBYTE                       ; GET ADDRESS
        STA ADH
        JSR RBYTE
        STA ADL
        LDX #$00
        LDA (ADL,X)                     ; AND IT'S DATA
        LDX #$06
        JSR WBYTE
        JSR GETBYT
        LDX CURCMD                      ; CHECK FOR +MEM/-MEM
        CPX #$3C
        BEQ SQA2
        INC ADL                         ; NEXT LOCATION
        BNE SEQADR
        INC ADH
        JMP SEQADR
SQA2:   DEC ADL                         ; PREVIOUS LOCATION
        LDA ADL
        CMP #$FF
        BNE SEQADR
        DEC ADH
        JMP SEQADR
;
;---------------------------------------
;   ### SHOW REGISTER-CONTENTS ###
;
; DISPLAYFORMAT: PPPP AA XX YY PS
; PPPP = PROGRAM COUNTER
; AA, XX, YY = A X Y REGISTERS
; PS = PROGRAM STATUS
;
RSHOW:  JSR CLRDSP
        LDX #$00
        LDA ADH
        JSR WBYTE
        LDA ADL
        JSR WBYTE
        INX
        LDA AC
        JSR WBYTE
        INX
        LDA XR
        JSR WBYTE
        INX
        LDA YR
        JSR WBYTE
        INX
        LDA PS
        JMP WBYTE
;
;---------------------------------------
;   ### INPUT AN ADDRESS ###
;
; READ AN ADDRESS UNTIL A <RETURN>
; 
INPADR: JSR WCOMND
        JSR GETADR
        JMP SQA1
;
;---------------------------------------
;   ### WRITE COMMANDCODE ONTO THE DISPLAY ###
;
WCOMND: JSR CLRDSP
WCOM1:  LDA CURCMD
        STA DIBUFF + 15
        RTS
;
;---------------------------------------
;   ### INSERT & DELETE ###
;
INSDEL: LDY #$00
ID0:    LDA EAL
        CMP ADL                         ; TEST FOR END
        LDA EAH
        SBC ADH
        BCS ID1
        BEQ ID1
        JMP SQA1                        ; OUT
ID1:    LDA CURCMD
        CMP #DELCMD
        BEQ IDD
        LDA (ADL),Y                     ; INSERT
        PHA
        TXA
        STA (ADL),Y
        PLA
        TAX
        JMP IDNEXT
IDD:    INY
        LDA (ADL),Y
        DEY
        STA (ADL),Y
IDNEXT: INC ADL
        BNE ID0
        INC ADH
        BNE ID0
;
;---------------------------------------
;   ### CALCULATE BRANCH OFFSET ###
;
; WHEN THE OFFSET IS OUT OF RANGE, '$$' IS
; WRITTEN INTO THE DATA-FIELD AND THE
; CORRECT ADDRESS IS TO BE TYPED IN.
;
OFFSET: JSR WCOMND
OFFS1:  JSR GETADR
        LDX #$02
        JSR RBYTE
        CLC
        SBC ADL
        PHP                             ; SAVE .C
        LDX #$00
        STA (ADL,X)
        JSR RBYTE                       ; CHECK FOR OFFSET IN RANGE
        PLP
        SBC ADH
        BEQ OFFS9                       ; OK.
        CMP #$FF                        ; NEG. OFFSET ?
        BEQ OFFS9                       ; OK.
        LDX #$06
        LDA #$13                        ; '--'BYTE.
        JSR WCHAR
        INX
        JSR WCHAR
        JMP OFFS1
OFFS9:  JMP SEQADR
;
;---------------------------------------
;   ### SINGLE STEP THE USER ###
;
; START THE SINGLE-STEP TIMER (T2) AND
; PERFORM A 'GO'-COMMAND.
;
SSUSER: LDA VECTAB + 2                  ; SET INT-VECTOR ON 'SYSTEM-INT'
        STA INTVEC
        LDA VECTAB + 3
        STA INTVEC + 1
        JSR BLKDIS
        JSR RELDIS
        LDA IER1
        ORA #$A0
        STA IER1
        LDA ACR1
        AND #$DF
        STA ACR1
        LDA #$24                        ; GENERATE 'IRQ' ON OPCODE-FETCH OF USER OPCODE
        STA T2LL1
        LDA #$00
        STA T2CH1
;
;---------------------------------------
;   ### GO TO THE USER-PROGRAM ###
;
; RESTORE THE USER REGISTERS FROM RAM INTO THE
; PROCESSOR. THEN CALL THE USER PROGRAM BY 'RTI'.
;
GOUSER: LDX SPTR                         ; DEFINE STACK
        TXS
        LDA ADH
        PHA
        LDA ADL
        PHA
        LDA PS
        PHA
        LDA AC
        LDX XR
        LDY YR
NSUP:   RTI
;
;---------------------------------------
;   ### ENTER MONITOR FROM "BRK" ###
;
; WHEN A 'BRK'-INSTRUCTION IS EXECUTED, ALL REGISTERS  
; MUST BE STORED. THE MONITOR-PROGRAM RESUMES EXECUTION
; TO ALLOW THE OPERATOR TO VIEW MEMORY AND REGISTERS.
;
BKUSER: STA AC
        STX XR
        STY YR
        PLA
        STA PS                         ; STATUS
        CLD
        PLA
        SEC
        SBC #$01                        ; CORRECT : FROM 'BRK'
        STA ADL                         ; PC
        PLA
        SBC #$00
        STA ADH
        TSX
        STX SPTR
        LDA T2LL1                       ; CLEAR TIMER2
        CLI
        JMP SEQADR                      ; SHOW NEXT LOCATION
;
;---------------------------------------
;   ### DISASSEMBLE AN INSTRUCTION ###
;
DISA:   JSR CLRDSP
        JSR DISASM
        SEC
        ADC ADL
        STA ADL
        BCC DISA1
        INC ADH
DISA1:  JSR GETUKY
        CMP #DISCMD
        BEQ DISA
        JMP JUMPER
;
;---------------------------------------
;   ### GET UPPER-CASE KEY ###
;
; TRANSFORM ALL KEYBOARD INPUT FOR UPPER-CASE
; CHARACTERS. THIS SIMPLIFIES THE MONITOR PROGRAM.
;
GETUKY: JSR GETKEY
        CMP #$60                        ; TRANSFORM LOWERCASE TO UPPERCASE
        BCC GUKY9
        CMP #$7B
        BCS GUKY9
        AND #$DF
GUKY9:  RTS
;
;---------------------------------------
;   ### GET A KEY FROM KEYBOARD ###
;
; THE GETKEY WILL PERFORM BOTH DOWN- & UP-DEBOUNCE
; ON ANY CHAR-KEY. THE SHIFT, ALPHA-LOCK AND 'CTRL'
; FUNCTIONS ARE PERFORMED LATER.
;
GETKEY: TXA
        PHA
        TYA
        PHA
        JSR ROVER
        BCS GKEY9
GKEY1:  JSR SETDBN
        JSR DECKEY
        BCC GKEY1
GKEY3:  JSR DHOLD
        BNE GKEY1
        JSR QDEBNC
        BNE GKEY3
        JSR SETDBN
        LDA #$1B
        STA REPTO
        JSR KDECOD
GKEY9:  PLA
        TAY
        PLA
        TAX
        LDA LSTKEY
        RTS
;
;---------------------------------------
;   ### CHECK FOR KEY-ENTRY ###
;
; EXIT .C=0 --> NO VALID CHR, NO KEY
; EXIT .C=1 --> VALID CHR IN .A
;
SCNKEY: TXA                             ; SAVE .I & .Y
        PHA
        TYA
        PHA
        JSR ROVER                       ; ROLL-OVER & REPEAT
        BCS SCNK9                       ; HAVE A REPEAT.
SCNK1:  JSR SETDBN                      ; START DEBOUNCE TIMER
SCNK2:  JSR DECKEY                      ; DECODE THE KEY(S)
        BCC SCNK9                       ; NO KEY DEPRESSED, EXIT .C=0
SCNK3:  JSR DHOLD                       ; STILL DOWN ??
        BNE SCNK1                       ; NO, RELEASE PREMATURELY
        JSR QDEBNC                      ; TIME-OUT ?
        BNE SCNK3                       ; NO . . 
        JSR SETDBN                      ; START TIMER FOR REPEAT
        LDA #$1B                        ; START REPEAT AFTER 0.8 SEC
        STA REPTO
        JSR KDECOD                      ; DECODE KEY WITH SHIFT & CNTL
SCNK9:  PLA                             ; RESTORE .Y & .X
        TAY
        PLA
        TAX
        LDA LSTKEY
        RTS
;
;---------------------------------------
;   ### KEY RELEASE ###
;
; WAIT TILL KEY IS RELEASED (WITH DEBOUNCE)
; AND GENERATE REPEAT WHEN HOLD (16 PER SEC).
;
ROVER:  JSR DHOLD                       ; STILL DOWN ??
        BNE ROVER5                      ; NO.
        JSR QDEBNC                      ; TIME-OUT
        BNE ROVER                       ; NO. TRY AGAIN
        JSR SETDBN                      ; RESTART TIMER
        DEC REPTO                       ; COUNT INTERVALS (200MS)
        BNE ROVER
        LDA #$03                        ; 60MS
        STA REPTO
        SEC                             ; REPEAT
        RTS
ROVER5: JSR SETDBN                      ; START TIMER
ROVER6: JSR DHOLD                       ; NOW SEE IF THE KEY
        BEQ ROVER                       ; NOT RELEASED.
        JSR QDEBNC                      ; TIME-OUT ??
        BNE ROVER6
        CLC
        RTS                             ; KEY IS RELEASED
DHOLD:  LDA IMASK
        BIT INKB                        ; .NE=RELEASED
        RTS
;
;---------------------------------------
;   ### START DEBOUNCE COUNTER ###
;
SETDBN: LDA #$03                        ; 20-30 MS
        STA DBCNTR                      ; START DEBOUNCE-TIMER
        RTS
;
;---------------------------------------
;   ### TEST FOR DEBOUNCE-TIME-OUT ###
;
QDEBNC: LDA DBCNTR
        RTS                             ; .EQ = TIME-OUT
;
;---------------------------------------
;   ### DECODE A KEY ###
;
; THIS ROUTINE DOES 2 THINGS:
; 1. SCAN THE 'SPECIAL' KEYS:
;    <L-SHIFT>, <R-SHIFT>, <ALPHA-LOCK>, <CNTL>
;    AND SAVE THESE BIT IN 'CPATRN'.
;    BIT 4: <ALPHA-LOCK>
;    BIT 5: <R-SHIFT>
;    BIT 6: <L-SHIFT>
;    BIT 7: <CNTL>
; 2. SCAN THE REST OF THE MATRIX AND HOLD THE OUTPUT
;    WHEN A 'CLOSED-CONTACT' IS DETECTED. (.C=1)
;    WHEN NO KEY IS FOUND THE .C=0
;
DECKEY: LDA #$01                        ; INIT MASKS
        STA IMASK                       ; (POSITIVE LOGIC)
        STA $0131
        LDX #$04                        ; WE HAVE 4 KEYS TO READ FIRST
DECK1:  JSR QKB                         ; SETOUT DMASK & READ INPUTS
        SEC
        BNE DECK2
        CLC
DECK2:  ROR CPATRN
        DEX
        BNE DECK1
DECK3:  LDA IMASK                       ; CHECK FOR END 
        AND OMASK
        AND #$01
        CLC
        BNE DECK9                       ; NO KEY DEPRESSED
        JSR QKB                         ; READ A MATRIX-KNOD
        BNE DECK3                       ; NOT CLOSED
        SEC
DECK9:  RTS
;
;---------------------------------------
;   ### READ A MATRIX-KNOD ###
;
QKB:    LSR OMASK                       ; INCR. MASKS
        BNE QKB1
        ROR OMASK                       ; .C=1
        LSR IMASK
        BNE QKB1
        ROR IMASK                       ; .C=1
QKB1:   LDA OMASK
; -INV- EOR #$FF                        ; <<< FOR NON-INV. OUTPUT
        STA OUTKB
        JSR QKB2                        ; DELAY 12 USEC
        LDA IMASK
        AND INKB
QKB2:   RTS
;
;---------------------------------------
;   ### DECODE 'KEYNR' FROM MATRIX(X,Y) ###
;
KDECOD: LDA IMASK
        LDX #$00
        JSR BINHEX                      ; CONVERT BIT-PATTERN TO HEX
        TXA
        ASL A                           ; ADJUST FOR O-MASK
        ASL A
        ASL A
        TAX
        LDA OMASK
        JSR BINHEX
        LDA KMATRX,X                     ; GET LOGICAL NUMBER
        STA KEYNR                       ; SAVE THE KEY'S NUMBER
;
; HAVE A SHIFT ??
;
        LDA CPATRN
        EOR #$60
        AND #$60
        BEQ KDEC5                       ; NO SHIFT
        LDX #SHIFT
        JSR SPROC                       ; CALL THE SHIFT-PROCESSOR
        JMP KDEC8
KDEC5:  LDA CPATRN
        AND #$10
        BNE KDEC7                       ; NO ALPHA-LOCK
        LDX #LOCK
        JSR SPROC                       ; CALL THE SHIFT-PROCESSOR
        JMP KDEC8
KDEC7:  LDA #$00                        ; GET VALUE FROM KEYNR
        JSR SPROC7
;
; NOW MODIFY THE CODE ON 'CTRL'
;
KDEC8:  LDA CPATRN
        BMI KDEC9                       ; NO CTRL
        LDA LSTKEY
        AND #%00011111
        STA LSTKEY
KDEC9:  LDA LSTKEY
        RTS
;
;---------------------------------------
;   ### CONVERT BIT (.A) TO HEX ADD TO .X ###
;
BINHEX: LSR A
        BCS BHEX9
        INX
        BNE BINHEX
BHEX9:  RTS
;
;---------------------------------------
;   ### CONVERT MATRIX TO KEY-NUMBERS ###
;
KMATRX: .DB $01, $2E, $38, $23, $03, $2D, $01, $35
        .DB $2C, $13, $29, $17, $05, $04, $25, $2A
        .DB $15, $16, $24, $26, $07, $06, $18, $28
        .DB $14, $19, $2B, $27, $09, $08, $1A, $20
        .DB $1F, $1C, $1B, $21, $0B, $0A, $1D, $0E
        .DB $10, $1E, $22, $0F, $0C, $02, $0D, $11
        .DB $01, $34, $37, $00, $12, $30, $31, $33
        .DB $01, $36, $2F, $32
; -- ALL UNASSIGNED KEYS ARE CODED AS THE SPACE-BAR --
;
;---------------------------------------
;   ### THIS TABLE CONTAINS INFORMATION OVER  ###
;       WHAT OPERATION MUST BE PERFORMED ONTO
;       WHITCH KEY.
;
;
LOCK    .EQU 0
;
SHFTBL: .DB $13, $2C                    ; KEYS 19 TILL 44
        .DB $20                         ; EOR #$20
        .DB $00
;        
SHIFT   .EQU $ - SHFTBL  
;
        .DB $03, $11                    ; KEYS 3 TILL 17
        .DB $10                         ; EOR #$10
        .DB $12, $30                    ; AND KEYS 18 TILL 48
        .DB $20                         ; EOR #$20
        .DB $00    
;
;---------------------------------------
;   ### SHIFT-PROCESSOR ###
;
; PERFORMS THE SCANNING OF THE LIST (.X)
; CHECKS IF KEYNR IS IN RANGE, GETS THE
; ASCII VALUE AND DOES THE OPERATION.
;
SPROC:  LDA SHFTBL,X
        BEQ SPROC7                      ; END OF LIST, NO-OP
        INX
        CMP KEYNR
        BEQ SPROC3
        BCS SPROC1                      ; NOT IN RANGE
SPROC3: LDA SHFTBL,X
        CMP KEYNR
        BCS SPROC2                      ; IN RANGE
SPROC1: INX
        INX                             ; TRY ON THE NEXT BLOCK
        BNE SPROC
SPROC2: INX
        LDA SHFTBL,X                    ; POINT TO THE OPERAND
SPROC7: LDY KEYNR
        EOR ASCVAL,Y
        STA LSTKEY                       ; SAVE ASCII VALUE
        RTS
;
;---------------------------------------
;   ### TRANSLATION OF KEYNR TO ASCII VALUE ###
;
ASCVAL: .DB $11                         ; CHAR INS
        .DB $20                         ; SPACE
        .DB "0123456789:;,-./"          ; KEY 2 - 17
        .DB "@abcdefghijklmno"          
        .DB "pqrstuvwxyz["
		.DB	$$5C
		.DB	"]^"           ; KEY 18 - 48
        .DB $12                         ; CHAR DEL
        .DB $0D                         ; KEY 50
        .DB $08
        .DB $0A
        .DB $0B
        .DB $0C
        .DB $1E                         ; HOME KEY 55
        .DB $1B                         ; ESC
;
;---------------------------------------
;   ### GET HEX KEY OR DO COMMAND ###
;
GETHEX: JSR GETUKY
        CMP #$0D
        BEQ GHEX9
        CMP #$30
        BCC GHEX5
        CMP #$3A
        BCC GHEX1
        CMP #$41
        BCC GHEX5
        CMP #$47
        BCS GHEX5
        SBC #$06                        ; .C=0
GHEX1:  AND #$0F
        CMP #$10                        ; ALWAYS .NE
GHEX9:  RTS
GHEX5:  JMP JUMPER                      ; GO TO EXECUTE THE COMMAND
;
;---------------------------------------
;   ### GET 4 DIGIT HEX ADDRESS ###
;
; GET A 4-DIGIT ADDRESS INTO THE ADDRESS-FIELD.
;
GETADR: LDA #$00
        TAX
GA0:    JSR WCHAR                       ; CLEAR ADR-FIELD
        INX
        CPX #$04
        BMI GA0
GA1:    JSR GETHEX
        BEQ GAEND
        LDX #$00
        PHA                             ; SAVE DATA
GA2:    LDA DIBUFF + 1,X                     ; LEFT-SHIFT 4 DIGITS
        STA DIBUFF,X
        INX
        CPX #$03
        BMI GA2
        PLA
        JSR WCHAR
        JMP GA1
GAEND:  RTS
;
;---------------------------------------
;   ### GET BYTE IN DATA FIELD ###
;
GETBYT: JSR GETHEX
        BEQ GBEND
        LDX DIBUFF + 7                       ; LEFT-SHIFT ON DATA-FIELD
        STX DIBUFF + 6
        LDX #$07
        JSR WCHAR
        JMP GETBYT
GBEND:  LDX #$06
        JSR RBYTE
        LDX #$00
        STA (ADL,X)
        RTS
;
;---------------------------------------
;   ### READ A BYTE FROM DISPLAY ###
;
; THE POSITION OF THE NUMBER READ IS INDEXED BY .X
; THE NUMBER IS READ AND IS LEFT IN .A
;
RBYTE:  JSR RCHAR
        ASL A
        ASL A
        ASL A
        ASL A
        STA TEMP + 1
        INX
        JSR RCHAR
        ORA TEMP + 1
        INX
        RTS
;
;---------------------------------------
;   ### WRITE A BYTE TO DISPL(X) ###
;
; WRITE A BYTE FROM .A TO THE DISPLAYPOSITION
; INDEXED BY .X
;   
WBYTE:  STA TEMP + 1
        LSR A
        LSR A
        LSR A
        LSR A
        JSR WCHAR
        INX
        LDA TEMP + 1
        AND #$0F
        JSR WCHAR
        INX
        RTS
;
;---------------------------------------
;   ### CLEAR DISPLAY TO BLANKS ###
;
CLRDSP: LDA #' '
        LDX #DIBUFL - 1
FILDSP: STA DIBUFF,X
        DEX
        BPL FILDSP
        INX
        STX CURPOS
        RTS
;
;---------------------------------------
;   ### WRITE CHR TO DISPL(X) ###
;
; WRITE A HEX-CHAR. INTO THE DISPLAY POSITION INDEXED
; BY .X
;
WCHAR:  STY YSAV
        TAY
        TXA
        AND #$0F
        TAX
        LDA CHASET,Y
        STA DIBUFF,X
        TYA
        LDY YSAV
        RTS
;
;---------------------------------------
;   ### OUTPUT CHAR TO CURSOR ON DISPLAY ###
;
BLANK:  LDA #$20                        ; OUTPUT SPACE 
OUTALL: STX OXSAV                       ; OUTPUT CHAR IN .A
        LDX CURPOS
        CMP #$08                        ; COMPARE TO BACKSPACE
        BEQ OUTAL5
        STA DIBUFF,X
        PHA
        INX
        TXA
        AND #$0F
        STA CURPOS
        LDX OXSAV
        PLA
        RTS
OUTAL5: DEX
        BPL OUTAL6
        LDX #DIBUFL - 1
OUTAL6: LDA #$20
        STA DIBUFF,X
        STX CURPOS
        LDX OXSAV
        LDA #$08                        ; LOAD BACKSPACE
        RTS
;
;---------------------------------------
;   ### READ CHR FROM DISPL(X) ###
;
; READ A CHAR FROM DISPLAYPOSITION .X
; THE CHAR IS CONVERTED TO ITS BINARY VALUE.
;
RCHAR:  STX XSAV
        LDA DIBUFF,X
        LDX #NCHARS
RC1:    CMP CHASET,X
        BEQ RC2
        DEX
        BPL RC1
        INX
RC2:    TXA
        LDX XSAV
        RTS
;
;---------------------------------------
;   ### CHARACTER SET ###
;
;
CHASET: .DB "01234"
        .DB "56789"
        .DB "ABCDEF"
        .DB " .=-"

NCHARS .EQU $ - CHASET         
;
;---------------------------------------
;   ### PRINT A TEXT ON THE DISPLAY ###
;
; THE TEXT IS ADDRESSED BY AN ADDRESS IN THE .Y/.A
; REGISTER PAIR. .Y = HIGH-BYTE, .A = LOW-BYTE.
; THE END OF THE TEXT IS MARKED BY A '00'.
;
TPRINT: STA TEMP                         ; TEXT ADDRESS IN .YA
        STY TEMP + 1
        JSR CLRDSP
        LDY #$00
TPRNT1: LDA (TEMP),Y
        BEQ TPRNT9                      ; #EOT#
        STA DIBUFF,Y
        INY
        BNE TPRNT1
TPRNT9: RTS
;
;---------------------------------------
;   ### TEST THE KEYBOARD/DISPLAY & I/O ###
;
TESTER: LDA #$00                        ; INIT VIA
        STA DDRA1
        LDA #$FF
        STA DDRB1
        LDA #$EC                        ; CA2=0, CB2=1
        STA PCR1
        JSR CLRDSP
TEST1:  LDX #$00
TEST2:  JSR GETKEY
        CMP #ESC
        BEQ TEST3
        STA DIBUFF,X
        LDA PCR1                        ; TOGGLE CA2 & CB2
        EOR #$22
        STA PCR1
        INX
        CPX #DIBUFL
        BCS TEST1
        BCC TEST2
TEST3:  JSR VIAINI
TEST4:  LDA DRA1                        ; ECHO INPUTS TO OUTPUTS
        STA DRB1
        BCS TEST4                       ; ALWAYS
;
;---------------------------------------
;   ### HANDLER FOR IRQ ###
;
IRQHDL: STA TEMP
        PLA
        PHA
        AND #$10
        BNE BRKINT
        LDA IFR1
        AND #$20
        BNE STEPIN                      ; SST-INTERRUPT TIMER T2
        LDA TEMP
        JMP (IRQVEC)
BRKINT: LDA TEMP
        JMP BKUSER
STEPIN: LDA TEMP
        JMP SSTINT
;
;---------------------------------------
;   ### HANDLER FOR NMI & IRQ ###
;
NMI:    JMP (NMIVEC)
IRQ:    JMP (INTVEC)
;
;---------------------------------------
;   ### HANDLE THE INTERRUPTS ###
;
TERMNL: PHA
        TXA
        PHA
        TYA
        PHA
        JSR DISPL                       ; NEXT CHAR
        JSR TIMUPD                      ; UPDATE TIMERS
        PLA
        TAY
        PLA
        TAX
        PLA
        RTI
;
;---------------------------------------
;   ### UPDATE THE TIMERS ###
;
TIMUPD: DEC DIV1                       ; EVERY 512 USEC
        BPL TIMUP9
        LDA #$13                        ; 10.24 MS INTERVAL
        STA DIV1
        LDX #$12                        ; 8 TIMERS & DBCNTR
TIMUP1: CLD
        SEC
        LDA TIMER,X
        SBC #$01
        STA TIMER,X
        LDA TIMER + 1,X
        SBC #$00
        STA TIMER + 1,X
        BCS TIMUP2                      ; PAST '0000' ??
        LDA #$00
        STA TIMER,X
        STA TIMER + 1,X
TIMUP2: DEX
        DEX
        BPL TIMUP1
TIMUP9: RTS
;
;---------------------------------------
;   ### REFRESH THE DISPLAY ###
;
DISPL:  LDA DCPTR                       ; SHOW NEXT CHR.
        EOR #$80
        STA DCPTR
        BMI DISP9                       ; ILLEGAL CHR.
        TAX
        INX
        CPX #$10
        BCC DISP1
        LDX #$00
DISP1:  STX DCPTR                       ; SAVE THE PTR
        LDA DIBUFF,X
        AND #$7F
        CMP #$60
        BCC DISP2
        AND #$DF
DISP2:  CMP #$2E
        BEQ DISP3
        CMP #$2C
        BNE DISP4
DISP3:  PHA
        LDA #$20
        JSR DISP4
        JSR DISP5
        PLA
DISP4:  LDY #$08
DISP6:  ASL A
        TAX
        LDA #$01
        ROL A
        STA CHSEL 
        AND #$01
        STA CHSEL
        TXA
        DEY
        BNE DISP6
DISP5:  STY $0E04                       ; ERROR IN BINARY ? CHSEL = $E004
        NOP
		NOP
DISP9:	RTS
;
;---------------------------------------
;   ### DISABLE & ENABLE DISPLAY INTERRUPTS (NMI) ###
;
BLKDIS: LDA #$20
        STA NMICTR                       ; CONTROL REGISTER
        RTS
RELDIS: JSR BLKDIS
        PHP
        SEI
        LDA #$04
        STA CHSEL
        JSR RELDI1
        LDA #$00
        STA CHSEL
        JSR RELDI1
        LDA #$FF
        STA DCPTR
        JSR DISP4
        LDA #$00
        STA NMICTR
        PLP
        RTS
RELDI1: LDX #$08
        LDY #$00
RELDI9: DEY
        BNE RELDI9
        DEX
        BNE RELDI9
        RTS
;
;---------------------------------------
;   ### ? ###
;
.DB  $FF, $A0, $F6, $A9, $C3, $20, $73, $F5, $20, $2A 
.DB  $F3, $AD, $35, $01, $C9, $FF, $D0, $F6, $A0, $F6
.DB  $A9, $D4, $20, $73, $F5, $20, $2A, $F3, $AD, $35
.DB  $01, $D0, $F8, $60, 
.DB  "RELEASE SPEC-KEY", $00
.DB  "PUSH ALL SPECS.", $00
.DB  $FF, $FF, $FF, $FF
;
;---------------------------------------
;   ### CASSETTE DUMP ###
;
; DUMP MEMORY TO TAPE
; IN 'KIM-6502' FORMAT
;
DUMP:   JSR BLKDIS                      ; BLOCK DISPLAY
        JSR DMPOPM
        JSR DMPMEM
        JSR DMPCLS
        JSR RELDIS                      ; RELEASE DISPLAY
        JMP SEQADR
;
;---------------------------------------
;   ### OPEN FOR DUMP  ###
;
DMPOPM: JSR DMPINI
        JSR DMPHED
        RTS
;
;---------------------------------------
;   ### INIT HARDWARE FOR DUMP ###
;
DMPINI: LDA #$C0                        ; T1 FREE RUNNING, PB7 PULS
        STA ACR1
        JSR VIAINI
        LDA #$00
        STA CHKSML
        STA CHKSMH
        STA T1CH1
        RTS
VIAINI: LDA #$C0                        ; SET CB2 LOW
        STA PCR1
        RTS
;
;---------------------------------------
;   ### DUMP HEADER ###
;
DMPHED: LDX #$64
        STX TEMP
        LDA #$16
NXTSNC: JSR DMPACC                      ; PRE-AMBLE
        DEC TEMP
        BNE NXTSNC
        LDA #$2A
        JSR DMPACC                      ; TRIGGER CHR
        LDA ID
        JSR DMPBYT                      ; ID. NUMBER
        LDA ADL
        JSR ADDCK
        JSR DMPBYT                      ; START ADDRESS
        LDA ADH
        JSR ADDCK
        JSR DMPBYT
        RTS
;
;---------------------------------------
;   ### DUMP BODY (MEMORY) ###
;
DMPMEM: LDY #$00
        LDA (ADL),Y
        JSR ADDCK
        JSR DMPBYT
        INC ADL
        BNE DM1
        INC ADH
DM1:    SEC
        LDA EAL
        SBC ADL
        LDA EAH
        SBC ADH
        BCS DMPMEM
        RTS
;
;---------------------------------------
;   ### DUMP END-RECORD ###
;
DMPCLS: LDA #$2F
        JSR DMPACC
        LDA CHKSML
        JSR DMPBYT                      ; CHECKSUM
        LDA CHKSMH
        JSR DMPBYT
        LDA #$04
        JSR DMPACC                      ; 'EOT'
        JSR DMPACC
        RTS
;
;---------------------------------------
;   ### ADD BYTE TO CHECKSUM ###
;
ADDCK:  PHA
        CLC
        ADC CHKSML
        STA CHKSML
        LDA CHKSMH
        ADC #$00
        STA CHKSMH
        PLA
        RTS
;
;---------------------------------------
;   ### DUMP ONE BYTE ###
;
DMPBYT: JSR HIASCH                      ; HEX --> ASCII HIGH-BYTE
        JSR DMPACC
        JSR HIASCL                      ; HEX --> ASCII LOW-BYTE
        JSR DMPACC
        RTS
;
;---------------------------------------
;   ### DUMP ACCUMULATOR ###
;
DMPACC: LDX #$08
        PHA
        PHA
NXTBIT: CLC
        JSR CPULSE                      ; 3700 HZ
        PLA
        LSR A
        PHA
        JSR CPULSE                      ; 3700 OR 2400 HZ
        SEC
        JSR CPULSE                      ; 2400 HZ
        DEX
        BNE NXTBIT
        PLA
        PLA
        RTS
;
;---------------------------------------
;   ### MAKE A PULSE ###
;
CPULSE: LDA #$00                        ; BIT IS IN CARRY
        ROL A
        TAY
        LDA FREQ,Y
        STA T1LL1
        LDA NPLS,Y
        TAY
MAITPL: BIT IFR1
        BVC MAITPL
        LDA T1CL1                       ; CLEAR INT. FLAG
        DEY
        BNE MAITPL
        RTS
;
;---------------------------------------
;   ### FREQUENCY DATA ###
;
FREQ:   .DB $88
        .DB $CD
NPLS:   .DB $12                         ; NUMBER OF HALF-PULSES
        .DB $0C
;
;---------------------------------------
;   ### CONVERSION HEX --> ASCII ###
;
HIASCH: STA $F5                         ; ENTRY FOR HIGH-BYTE
        LSR A
        LSR A
        LSR A
        LSR A
        BPL HEXASC
HIASCL: LDA TEMP + 1                    ; ENTRY FOR LOW BYTE
        AND #$0F
HEXASC: CMP #$0A
        BCC HA1
        ADC #$06
HA1:    ADC #$30
        RTS
;
;---------------------------------------
;   ### CASSETTE LOAD ###
;
; LOAD MEMORY FROM TAPE
; IN 'KIM-6502' FORMAT
;
;
;---------------------------------------
;   ### LOAD MAIN ROUTINE ###
;
LOAD:   JSR BLKDIS                      ; BLOCK DISPLAY
        LDA #$E0                        ; CB2 HIGH
        STA PCR1
        LDA #$7F                        ; PB7 FOR INPUT
        STA DDRB1
        LDA #$00
        STA ACR1                       ; T2 ONE-SHOT
        STA CHKSML                       ; CLEAR CHECKSUM
        STA CHKSMH           
;
; ### READ THE SYNC-BYTE'S ###
LODSHC: JSR RDBIT
        ROR	A
        CMP #$16
        BNE LODSHC
        LDA #$09
        STA TEMP + 1
LS1:    JSR LDACC
        CMP #$16
        BNE LODSHC
        DEC TEMP + 1
        BPL LS1
;
; ### FIND TRIGGER-WORD ###
LODSTR: JSR LDACC
        CMP #$2A
        BEQ LS2
        CMP #$16
        BEQ LODSTR
        BNE LODSHC
LS2:    JSR RDBYTE
        STA TAPID                       ; LEAVE CURRENT ID.
        CMP ID
        BEQ LODSAD                      ; LOAD EVERY FILE
        LDA ID
        CMP #$00
        BEQ LODSAD
        CMP #$FF                        ; ID=$FF 'RELOCATE & LOAD'
        BNE LODSHC                      ; FIND NEXT FILE
        JSR RDBYTE
        JSR ADDCK
        JSR RDBYTE
        JSR ADDCK
        JMP LODDAT
;
; ### SET START ADDRESS ###
LODSAD: JSR RDBYTE
        JSR ADDCK
        STA ADL
        JSR RDBYTE
        JSR ADDCK
        STA ADH
;
; ### LOAD DATA TO MEM ###
LODDAT: JSR RDBYTE
        BCS LODEND                      ; #EOF#
        JSR ADDCK
        LDY #$00
        STA (ADL),Y
        INC ADL
        BNE LODDAT
        INC ADH
        BNE LODDAT
;
; ### TEST CHECKSUM ###
LODEND: JSR RDBYTE
        CMP CHKSML
        BNE LE1
        JSR RDBYTE
        CMP CHKSMH
        BEQ LE2
LE1:    LDA #$80
        ORA TAPID                       ; MARK TAPE-ERROR
        STA TAPID
        LDA #$45
        STA CURCMD
LE2:    LDA T2LL1                       ; CLEAR T2 INT. FOR 'SST'
        JSR RELDIS                      ; RELEASE DISPLAY
        JSR VIAINI
        JMP SEQADR
RDBYTE: JSR RDHEX
        BCS RDB1                        ; #EOF#
        ASL A
        ASL A
        ASL A
        ASL A
        STA TEMP
        JSR RDHEX
        BCS RDB1                        ; #EOF#
        ORA TEMP
        CLC
RDB1:   RTS
;
;---------------------------------------
;   ### READ A HEX NUMBER ###
;
RDHEX:  JSR LDACC
        CMP #$2F
        BEQ RH2                         ; #EOF# 
        SEC
        SBC #$30
        CMP #$0A
        BMI RH1
        SEC
        SBC #$07
        CLC                             ; RETURN WITH HEX NUMBER
RH1:    RTS
RH2:    SEC                             ; RETURN WITH #EOF# MARKER
        RTS
;
;---------------------------------------
;   ### READ 8 BIT'S TO ACC ###
;
LDACC:  LDA #$00
        LDX #$07
LDA1:   JSR RDBIT
        ROR A
        DEX
        BPL LDA1
        RTS
;
;---------------------------------------
;   ### READ ONE BIT ###
;
RDBIT:  PHA
RB0:    JSR GETFRQ
        BEQ RB0                         ; WAIT FOR 2400 HZ
        LDY #$00
RB1:    JSR GETFRQ
        BEQ RB2                         ; WAIT FOR 3700 HZ
        INY
        BPL RB1
RB2:    CPY #$09                        ; BIT IS IN CARRY
        PLA
        RTS
;
;---------------------------------------
;   ### TIME PULSE LENGTH ###
;
; THE COMPLETE PERIOD-TIME IS DETERMINED,
; SO THE LOGIC IS INSENSITIVE FOR PHASE-SHIFTS
;
GETFRQ: BIT DRB1
        BMI GETFRQ                      ; WAIT TILL LOW
        BIT DRB1                        ; DEBOUNCE
        BMI GETFRQ
GF0:    BIT DRB1
        BPL GF0                         ; WAIT TILL HIGH
        BIT DRB1                        ; DEBOUNCE
        BPL GF0
        LDA IFR1                        ; READ T2-STATUS OF LAST
        PHA                             ; PERIODE
        LDA #$3C                        ; AND RETRIGGER
        STA T2LL1
        LDA #$01
        STA T2CH1
        PLA
        AND #$20
        RTS
;
;---------------------------------------
;   ### DISASSEMBLER ###
;
DISASM:      JSR  lbl_subF9DC
             LDY #$00
             JSR lbl_subFF0B
             TAY
             LSR A
             BCC lbl_jmpF91A
             LSR A
             BCS lbl_jmpF929
             CMP #$22
             BEQ lbl_jmpF929
             AND #$07
             ORA #$80
lbl_jmpF91A: LSR A
             TAX
             LDA $F9FC,X
             BCS lbl_jmpF925
             LSR A
             LSR A
             LSR A
             LSR A
lbl_jmpF925: AND #$0F
             BNE lbl_jmpF92D
lbl_jmpF929: LDY #$80
             LDA #$00
lbl_jmpF92D: TAX
             LDA $FA40,X
             STA $015A
             AND #$03
             STA $0157
             TYA
             AND #$8F
             TAX
             TYA
             LDY #$03
             CPX #$8A
             BEQ lbl_jmpF94F
lbl_jmpF944: LSR A
             BCC lbl_jmpF94F
             LSR A
lbl_jmpF948: LSR A
             ORA #$20
             DEY
             BNE lbl_jmpF948
             INY
lbl_jmpF94F: DEY
             BNE lbl_jmpF944
             PHA
             JSR BLANK
             PLA
             TAY
             LDA $FA5A,Y
             STA $0159
             LDA $FA9A,Y
             STA $0158
             LDX #$03
lbl_jmpF966: LDA #$00
             LDY #$05
lbl_jmpF96A: ASL $0158
             ROL $0159
             ROL A
             DEY
             BNE lbl_jmpF96A
             ADC #$3F
             JSR OUTALL
             DEX
             BNE lbl_jmpF966
             JSR BLANK
             LDX #$06
             LDA #$00
             STA TEMP
lbl_jmpF985: CPX #$03
             BNE lbl_jmpF9A6
             LDY $0157
             BEQ lbl_jmpF9A6
lbl_jmpF98E: LDA $015A
             CMP #$E8
             JSR lbl_subFF0B
             BCS lbl_jmpF9C5
             PHA
             LDA TEMP
             BNE lbl_jmpF99F
             INC TEMP
lbl_jmpF99F: PLA
             JSR lbl_subFEF4
             DEY
             BNE lbl_jmpF98E
lbl_jmpF9A6: ASL $015A
             BCC lbl_jmpF9B7
             LDA $FA4D,X
             JSR lbl_subF9BD
             LDA $FA53,X
             JSR lbl_subF9BD
lbl_jmpF9B7: DEX
             BNE lbl_jmpF985
             JMP lbl_jmpF9D0
lbl_subF9BD: CMP #$2F
             BEQ lbl_jmpF9C4
             JSR OUTALL
lbl_jmpF9C4: RTS
;
;---------------------------------------
;   ###  ###
;
lbl_jmpF9C5: JSR lbl_subF9F0
             TAX
             INX
             BNE lbl_jmpF9CD
             INY
lbl_jmpF9CD: JSR lbl_subF9D4
lbl_jmpF9D0: LDA $0157
             RTS
;
;---------------------------------------
;   ###  ###
;
lbl_subF9D4: TYA
lbl_jmpF9D5: JSR lbl_subFEF4
             TXA
             JMP lbl_subFEF4
lbl_subF9DC: LDA ADH
             LDX ADL
             JMP lbl_jmpF9D5
;
;---------------------------------------
;   ###  ###
;
.DB $A2, $03, $20, $1A, $F5, $CA, $D0, $FA, $60, $AD, $57, $01, $38    ; ASCII: ........`.W.8
;
;---------------------------------------
;   ###  ###
;
lbl_subF9F0: LDY ADH
             TAX
             BPL lbl_jmpF9F6
             DEY
lbl_jmpF9F6: ADC ADL
             BCC lbl_jmpF9FB
             INY
lbl_jmpF9FB: RTS
;
;---------------------------------------
;   ###  ###
;
	.DB  $40, $02, $45, $03, $D0, $08, $40, $09, $30, $22, $45, $33, $D0, $08, $40, $09, $40, $02, $45, $33, $D0, $08, $40, $09 
	.DB  $40, $02, $45, $B3, $D0, $08, $40, $09, $00, $22, $44, $33, $D0, $8C, $44, $00, $11, $22, $44, $33, $D0, $8C, $44, $9A 
	.DB	 $10, $22, $44, $33, $D0, $08, $40, $09, $10, $22, $44, $33, $D0, $08, $40, $09, $62, $13, $78, $A9, $00, $21, $01, $02 
	.DB  $00, $80, $59, $4D, $11, $12, $06, $4A, $05, $1D, $3B, $29, $3B, $23, $28, $2F, $59, $2F, $58, $2F, $2F, $41, $1C, $8A
	.DB  $1C, $23, $5D, $8B, $1B, $A1, $9D, $8A, $1D, $23, $9D, $8B, $1D, $A1, $00, $29, $19, $AE, $69, $A8, $19, $23, $24, $53, $1B
	.DB  $23, $24, $53, $19, $A1, $00, $1A, $5B, $5B, $A5, $69, $24, $24, $AE, $AE, $A8, $AD, $29, $00, $7C, $00, $15, $9C, $6D
	.DB  $9C, $A5, $69, $29, $53, $84, $13, $34, $11, $A5, $69, $23, $A0, $D8, $62, $5A, $48, $26, $62, $94, $88, $54, $44, $C8
	.DB	 $54, $68, $44, $E8, $94, $00, $B4, $08, $84, $74, $B4, $28, $6E, $74, $F4, $CC, $4A, $72, $F2, $A4, $8A, $00, $AA, $A2
	.DB  $A2, $74, $74, $74, $72, $44, $68, $B2, $32, $B2, $00, $22, $00, $1A, $1A, $26, $26, $72, $72, $88, $C8, $C4, $CA, $26
	.DB	 $48, $44, $44, $A2, $C8, $00, $02, $00, $08, $F2, $FF, $80, $01, $C0, $E2, $C0, $C0, $FF, $00, $00, $08, $00, $10, $80
	.DB	 $40, $C0, $00, $C0, $00, $40, $00, $00, $E4, $20, $80, $00, $FC, $00, $08, $08, $F8, $FC, $F4, $0C, $10, $04, $F4, $00
	.DB  $20, $10, $00, $00, $0F, $01, $01, $01, $11, $11, $02, $02, $11, $11, $02, $12, $02, $00, $08, $10, $18, $20, $28, $30
	.DB	 $38, $40, $48, $50, $58, $60, $68, $70, $78, $80, $88, $90, $98, $AC, $A8, $B0, $B8, $CC, $C8, $D0, $D8, $EC, $E8, $F0
	.DB  $F8, $0C, $2C, $4C, $4C, $8C, $AC, $CC, $EC, $8A, $9A, $AA, $BA, $CA, $DA, $EA, $FA, $0E, $2E, $4E, $6E, $8E, $AE, $CE
	.DB  $EE, $0D, $2D, $4D, $6D, $8D, $AD, $CD, $ED, $0D, $0D, $0C, $0D, $0E, $0D, $0C, $0D, $0D, $0D, $0C, $0D, $0D, $0D, $0C
	.DB  $0D, $0F, $0D, $0C, $0D, $09, $0D, $0C, $0D, $08, $0D, $0C, $0D, $08, $0D, $0C, $0D, $0F, $06, $0B, $0B, $04, $0A, $08
	.DB  $08, $0D, $0D, $0D, $0D, $0D, $0F, $0D, $0F, $07, $07, $07, $07, $05, $09, $03, $03, $01, $01, $01, $01, $02, $01, $01, $01    
;
;---------------------------------------
;   ### DISASSEMBLER ###
;
lbl_jmpFB96: LDA $F8
             STA $EE
             LDA $F9
             STA $EF
lbl_jmpFB9E: LDY #$00
             JSR lbl_subFEE5
             JSR CLRDSP
             LDY #$40
             JSR lbl_subFEE5
             LDX #$00
             STX $0156
             JSR lbl_subF9DC
             JSR BLANK
             JMP lbl_jmpFE0C
lbl_jmpFBB9: LDA #$00
             STA $0139
             STA $013A
             JSR BLANK
             LDY $013B
             SEC
lbl_jmpFBC8: ROR $0139
             ROR $013A
             DEY
             BNE lbl_jmpFBC8
             LDY $013B
             CPY #$0D
             BNE lbl_jmpFBDD
             LDX #$00
             JMP lbl_jmpFCC3
lbl_jmpFBDD: LDY #$06
             LDA #$51
lbl_jmpFBE1: STA $013E,Y
             DEY
             BNE lbl_jmpFBE1
             JSR lbl_subFE9D
             CMP #$20
             BEQ lbl_jmpFBDD
lbl_jmpFBEE: STA $013F,Y
             INY
             CPY #$07
             BCS lbl_jmpFC52
             JSR lbl_subFE9D
             CMP #$20
             BNE lbl_jmpFC02
             INC $0156
             BNE lbl_jmpFC06
lbl_jmpFC02: CMP #$0D
             BNE lbl_jmpFBEE
lbl_jmpFC06: STY $011A
             LDA $013F
             CMP #$23
             BEQ lbl_jmpFC35
             CMP #$28
             BEQ lbl_jmpFC6E
             LDA $011A
             CMP #$01
             BNE lbl_jmpFC20
             LDX #$01
             JMP lbl_jmpFCC3
lbl_jmpFC20: CMP #$02
             BNE lbl_jmpFC38
             LDA $013B
             CMP #$0C
             BNE lbl_jmpFC30
             LDX #$02
             JMP lbl_jmpFCC3
lbl_jmpFC30: LDX #$05
             JMP lbl_jmpFCC3
lbl_jmpFC35: JMP lbl_jmpFCAE
lbl_jmpFC38: LDA #$04
             CMP $011A
             BCC lbl_jmpFC54
             LDX #$02
             JSR lbl_subFDF7
             BNE lbl_jmpFC9E
             BCC lbl_jmpFC4D
             LDX #$03
             JMP lbl_jmpFCC3
lbl_jmpFC4D: LDX #$04
             JMP lbl_jmpFCC3
lbl_jmpFC52: BCS lbl_jmpFCBD
lbl_jmpFC54: JSR lbl_subFDF5
             BNE lbl_jmpFCBD
             BCC lbl_jmpFC6A
             LDA #$09
             CMP $013B
             BNE lbl_jmpFC66
             LDX #$0E
             BNE lbl_jmpFCC3
lbl_jmpFC66: LDX #$08
             BNE lbl_jmpFCC3
lbl_jmpFC6A: LDX #$09
             BNE lbl_jmpFCC3
lbl_jmpFC6E: LDA $0142
             CMP #$2C
             BEQ lbl_jmpFC79
             CMP #$58
             BNE lbl_jmpFC7D
lbl_jmpFC79: LDX #$0B
             BNE lbl_jmpFCC3
lbl_jmpFC7D: CMP #$29
             BNE lbl_jmpFC8C
             JSR lbl_subFDF5
             BNE lbl_jmpFCBD
             BCC lbl_jmpFCBD
             LDX #$0A
             BNE lbl_jmpFCC3
lbl_jmpFC8C: LDA $0144
             CMP #$29
             BNE lbl_jmpFCBD
             LDA $013B
             CMP #$0B
             BNE lbl_jmpFCBD
             LDX #$0D
             BNE lbl_jmpFCC3
lbl_jmpFC9E: LDA $013B
             CMP #$0C
             BNE lbl_jmpFCAA
             LDX #$02
             JMP lbl_jmpFCC3
lbl_jmpFCAA: LDX #$0C
             BNE lbl_jmpFCC3
lbl_jmpFCAE: LDA $013B
             CMP #$01
             BEQ lbl_jmpFCB9
             LDX #$07
             BNE lbl_jmpFCC3
lbl_jmpFCB9: LDX #$06
             BNE lbl_jmpFCC3
lbl_jmpFCBD: JSR lbl_subFEDC
             JMP lbl_jmpFB9E
lbl_jmpFCC3: LDA $FADA,X
             BEQ lbl_jmpFCCD
             AND $0139
             BNE lbl_jmpFCD5
lbl_jmpFCCD: LDA $FAE9,X
             AND $013A
             BEQ lbl_jmpFCBD
lbl_jmpFCD5: CLC
             LDA $FAF8,X
             ADC $0153
             STA $0153
             LDA $FB07,X
             BEQ lbl_jmpFD31
             CMP #$0F
             BEQ lbl_jmpFD05
             STA $0119
             AND #$0F
             TAY
             STA $011B
             INC $011B
             LDA $0119
             AND #$F0
             LSR A
             LSR A
             LSR A
             LSR A
             TAX
             JSR lbl_subFD08
             BCS lbl_jmpFCBD
             BCC lbl_jmpFD22
lbl_jmpFD05: JMP lbl_jmpFD90
lbl_subFD08: LDA $013F,X
             JSR lbl_subFEAE
             BCS lbl_jmpFD21
             INX
             LDA $013F,X
             INX
             JSR lbl_subFEB4
             BCS lbl_jmpFD21
             STA $0153,Y
             DEY
             BNE lbl_subFD08
             CLC
lbl_jmpFD21: RTS
lbl_jmpFD22: LDY $011B
lbl_jmpFD25: DEY
             LDA $0153,Y
             STA ($EE),Y
             CPY #$00
             BEQ lbl_jmpFD38
             BNE lbl_jmpFD25
lbl_jmpFD31: LDA #$01
             STA $011B
             BNE lbl_jmpFD22
lbl_jmpFD38: JSR CLRDSP
             JSR lbl_subFD87
             JSR DISASM
             LDA $0156
             BEQ lbl_jmpFD70
             LDY #$64
             JSR lbl_subFEE5
             JSR CLRDSP
             JSR lbl_subF9DC
             JSR BLANK
             JSR BLANK
             LDX $011B
             LDY #$00
lbl_jmpFD5C: LDA ($EE),Y
             JSR lbl_subFEF4
             JSR BLANK
             INY
             DEX
             BNE lbl_jmpFD5C
             LDY #$00
             JSR lbl_subFEE5
             JSR lbl_subFEE5
lbl_jmpFD70: LDY $011B
             JSR lbl_subFD7C
             JSR lbl_subFD87
             JMP lbl_jmpFB9E
lbl_subFD7C: TYA
             CLC
             ADC $EE
             STA $EE
             BCC lbl_jmpFD86
             INC $EF
lbl_jmpFD86: RTS
lbl_subFD87: LDA $EE
             STA $F8
             LDA $EF
             STA $F9
             RTS
lbl_jmpFD90: LDA $011A
             CMP #$02
             BNE lbl_jmpFDA8
             LDX #$00
             LDY #$01
             JSR lbl_subFD08
             BCS lbl_jmpFDDE
             LDA #$02
             STA $011B
             JMP lbl_jmpFD22
lbl_jmpFDA8: LDX #$00
             LDY #$02
             JSR lbl_subFD08
             BCS lbl_jmpFDDE
             LDA $EF
             STA $013A
             LDA $EE
             CLC
             ADC #$02
             STA $0139
             BCC lbl_jmpFDC3
             INC $013A
lbl_jmpFDC3: SEC
             LDA $0154
             SBC $0139
             STA $0154
             LDA $0155
             SBC $013A
             STA $0155
             CMP #$00
             BEQ lbl_jmpFDE8
             CMP #$FF
             BEQ lbl_jmpFDE1
lbl_jmpFDDE: JMP lbl_jmpFCBD
lbl_jmpFDE1: LDA $0154
             BMI lbl_jmpFDED
             BPL lbl_jmpFDDE
lbl_jmpFDE8: LDA $0154
             BMI lbl_jmpFDDE
lbl_jmpFDED: LDA #$02
             STA $011B
             JMP lbl_jmpFD22
lbl_subFDF5: LDX #$04
lbl_subFDF7: LDA $013F,X
             CMP #$2C
             BNE lbl_jmpFE02
             INX
             LDA $013F,X
lbl_jmpFE02: CMP #$58
             BEQ lbl_jmpFE09
             CMP #$59
lbl_jmpFE08: RTS
lbl_jmpFE09: CLC
             BCC lbl_jmpFE08
lbl_jmpFE0C: LDY #$00
             STY $0153
             STY $0154
             STY $0155
             STY $0139
lbl_jmpFE1A: JSR lbl_subFE9D
             CMP #$2A
             BEQ lbl_jmpFE74
             CMP #$20
             BEQ lbl_jmpFE1A
             AND #$1F
             STA $013C,Y
             TYA
             TAX
             INC $013C,X
             INY
             CPY #$03
             BNE lbl_jmpFE1A
             LDY #$03
lbl_jmpFE36: LDA $013B,Y
             LDX #$05
lbl_jmpFE3B: LSR A
             ROR $0139
             ROR $013A
             DEX
             BNE lbl_jmpFE3B
             DEY
             BNE lbl_jmpFE36
             LDX #$40
lbl_jmpFE4A: LDA $0139
lbl_jmpFE4D: CMP $FA59,X
             BEQ lbl_jmpFE57
             DEX
             BNE lbl_jmpFE4D
             BEQ lbl_jmpFE62
lbl_jmpFE57: LDA $013A
             CMP $FA99,X
             BEQ lbl_jmpFE65
             DEX
             BNE lbl_jmpFE4A
lbl_jmpFE62: JMP lbl_jmpFCBD
lbl_jmpFE65: LDA $FB55,X
             STA $013B
             LDA $FB15,X
             STA $0153
             JMP lbl_jmpFBB9
lbl_jmpFE74: LDA #$8D
             LDY #$FE
             JSR TPRINT
             JSR GETADR
             LDX #$00
             JSR RBYTE
             STA $F9
             JSR RBYTE
             STA $F8
             JMP lbl_jmpFB96
;
;---------------------------------------
;   ###  ###
;
.DB  $20, $20, $20, $20, $3D, $50, $43, $00    ; ASCII: ....=PC.
;
;---------------------------------------
;   ###  ###
;
lbl_jmpFE95: CPY #$00
             BEQ lbl_subFE9D
             JSR OUTALL
             DEY
lbl_subFE9D: JSR GETUKY
             CMP #$1B
             BEQ lbl_jmpFEAB
             CMP #$08
             BEQ lbl_jmpFE95
             JMP OUTALL
lbl_jmpFEAB: JMP MONITR
lbl_subFEAE: PHA
             LDA #$00
             STA $F4
             PLA
lbl_subFEB4: CMP #$30
             BCC lbl_jmpFEDA
             CMP #$47
             BCS lbl_jmpFEDA
             CMP #$3A
             BCC lbl_jmpFEC6
             CMP #$40
             BCC lbl_jmpFEDA
             ADC #$08
lbl_jmpFEC6: ROL A
             ROL A
             ROL A
             ROL A
             STX $F5
             LDX #$04
lbl_jmpFECE: ROL A
             ROL $F4
             DEX
             BNE lbl_jmpFECE
             LDX $F5
             LDA $F4
             CLC
             RTS
lbl_jmpFEDA: SEC
             RTS
lbl_subFEDC: LDA #$EE
             LDY #$FE
             JSR TPRINT
             LDY #$00
lbl_subFEE5: LDX #$00
lbl_jmpFEE7: DEX
             BNE lbl_jmpFEE7
             DEY
             BNE lbl_jmpFEE7
             RTS
;
;---------------------------------------
;   ###  ###
;
.DB  $45, $52, $52, $4F, $52, $00    ; ASCII: ERROR.
;
;---------------------------------------
;   ###  ###
;
lbl_subFEF4: PHA
             LSR A
             LSR A
             LSR A
             LSR A
             JSR lbl_subFEFF
             PLA
             AND #$0F
lbl_subFEFF: CLC
             ADC #$30
             CMP #$3A
             BCC lbl_jmpFF08
             ADC #$06
lbl_jmpFF08: JMP OUTALL
lbl_subFF0B: LDA ($F8),Y
             RTS
;
;---------------------------------------
;   ###  ###
;
	.DB $FF, $FF, $A0, $FF, $A9, $3D, $20, $73, $F5, $20, $58, $FF, $20, $03, $F0, $20, $A9, $FF, $20, $C0, $FF, $E0, $00, $D0
	.DB	$F3, $20, $A2, $F6, $A0, $FF, $A9, $4D, $20, $73, $F5, $A9, $3C, $8D, $07, $01, $AD, $07, $01, $D0, $FB, $F0, $D3, $4B
	.DB $45, $59, $42, $4F, $41, $52, $44, $20, $54, $45, $53, $54, $45, $52, $00, $2D, $2D, $20, $44, $4F, $4E, $45, $20, $2D
	.DB $2D, $00, $A0, $45, $B9, $64, $FF, $99, $01, $00, $88, $10, $F7, $60, $03, $04, $05, $06, $07, $08, $09, $0A, $0B, $02 
	.DB $0F, $30, $2E, $2D, $2F, $09, $0A, $0B, $38, $23, $29, $17, $24, $26, $2B, $27, $1B, $21, $22, $12, $31, $00, $32, $06
	.DB $07, $08, $13, $25, $16, $18, $19, $1A, $1C, $1D, $1E, $0D, $0C, $37, $35, $34, $03, $04, $05, $2C, $2A, $15, $28, $14
	.DB $20, $1F, $0E, $10, $11, $33, $36, $0F, $02, $10, $01, $AD, $33, $01, $A0, $00, $D9, $01, $00, $F0, $07, $C8, $C0, $45
	.DB $90, $F6, $B0, $05, $A9, $FF, $99, $01, $00, $60, $20, $F8, $F4, $A0, $00, $A2, $00, $B9, $01, $00, $30, $08, $20, $DA
	.DB $FF, $E8, $E0, $10, $F0, $05, $C8, $C0, $45, $D0, $EE, $60, $84, $00, $A8, $B9, $37, $F4, $A4, $00, $9D, $20, $01, $60
	.DB $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF, $DD, $F5, $2A, $F0
	.DB $E0, $F5    

.end